function [ highOrLow confidence x y] = shlomiAltOpt(milpProblem,highRxns,lowRxns,model)
%[ highOrLow confidence x y] = shlomiAltOpt(milpProblem,highRxns,lowRxns,model)
%
% Chris Gowen
% April, 2012
%
% Determine the effects of alternate optima on the flux states of the
% reactions with high and low expression states.
% 
% INPUTS:
%    - milpProblem......MILP problem structure as constructed by
%                       Shlomi_method.m and for input to the cplexmilp 
%                       command.
%    - highRxns.........List of reaction indices for reactions having high
%                       expression states
%    - lowRxns..........List of reaction indices for reactions having low
%                       expression states
%    - model............Structure containing the metabolic model in
%                       COBRA-readable format
%
% OUTPUTS:
%    - highOrLow........Vector with entry for all reactions in the model,
%                       having discrete values representing flux state:
%                          1: Reaction flux is always high (or not low)
%                         -1: Reaction flux is always low (or not high)
%                          0: Reaction flux is undetermined
%    - confidence.......Confidence vector for each reaction
%    - x ...............Vector of maximum agreement when forcing a flux
%                       to agree with the expression state
%    - y ...............Vector of maximum agreement when forcing a flux
%                       to disagree with the expression state
% 
% This function is provided to accompany the chapter "Linking RNA 
% measurements and proteomics with genome-scale models" in Methods in
% Molecular Biology: Systems Metabolic Engineering, ed. Hal Alper
%
% It is provided for educational purposes only, without any warranty.

highOrLow = zeros(length(model.rxns),1);
x = zeros(length(model.rxns),1);
y = zeros(length(model.rxns),1);
confidence = zeros(length(model.rxns),1);
milpTemp = milpProblem;
cplexOptions = cplexoptimset('Threads',40,'MaxTime',3600);
milpTemp.options=cplexOptions;
for i=1:length(highRxns)
    % Force reaction to be on
    r=highRxns(i);
    j_f = length(model.rxns)+length(lowRxns)+i; % column index for yf_h
    j_r = length(model.rxns)+length(lowRxns)+length(highRxns)+i; % column index for yr_h
    milpTemp.Aeq = [milpProblem.Aeq; zeros(1,size(milpProblem.Aeq,2))];
    milpTemp.Aeq(end,j_f)=1;
    milpTemp.Aeq(end,j_r)=1;
    milpTemp.beq = [milpProblem.beq; 1];
    
    % Solve tmp MILP:
    [~, f_ exit_ output_] = cplexmilp(milpTemp);
    if (exit_==1)
        x(r) = -f_;
    elseif (output_.cplexstatus==103)
        x(r) = 0;
        sprintf('Forcing reaction %s high is infeasible',model.rxns{r})    
    else 
        warning(sprintf('Problem forcing reaction %s high: %s',model.rxns{r},output_.cplexstatusstring)); 
    end
    
    % Force reaction to be off
    milpTemp.beq(end) = 0;
    
    % Solve tmp MILP:
    [~, f_ exit_ output_] = cplexmilp(milpTemp);
    if (exit_==1)
        y(r) = -f_;
    elseif (output_.cplexstatus==103)
        y(r) = 0;
        sprintf('Forcing reaction %s "not high" is infeasible',model.rxns{r})     
    else 
        warning(sprintf('Problem forcing reaction %s "not high": %s',model.rxns{r},output_.cplexstatusstring)); 
    end
    
    % Evaluate direction (high=1, low=-1)
    if (x(r)>y(r))
        highOrLow(r) = 1;
        confidence(r) = x(r)-y(r);
        use = 'always high';
    elseif (x(r)<y(r))
        highOrLow(r) = -1;
        confidence(r) = y(r)-x(r);
        use = 'never high';
    else
        use = 'undetermined';
    end
    sprintf('Reaction %s found to be %s with confidence=%i',...
    model.rxns{r},use,confidence(r))
end

for i=1:length(lowRxns)
    % Force reaction to be low
    r=lowRxns(i);
    j_f = length(model.rxns)+length(lowRxns)+i; % column index for yf_l
    milpTemp.Aeq = [milpProblem.Aeq; zeros(1,size(milpProblem.Aeq,2))];
    milpTemp.Aeq(end,j_f)=1;
    milpTemp.beq = [milpProblem.beq; 1];
    
    % Solve tmp MILP:
    [~, f_ exit_ output_] = cplexmilp(milpTemp);
    if (exit_==1)
        x(r) = -f_;
    elseif (output_.cplexstatus==103)
        x(r) = 0;
        sprintf('Forcing reaction %s low is infeasible',model.rxns{r})
    else 
        warning(sprintf('Problem forcing reaction %s on: %s',model.rxns{r},output_.cplexstatusstring)); 
    end
    
    % Force reaction to be 'not low'
    milpTemp.beq(end) = 0;
    
    % Solve tmp MILP:
    [~, f_ exit_ output_] = cplexmilp(milpTemp);
    if (exit_==1)
        y(r) = -f_;
    elseif (output_.cplexstatus==103)
        y(r) = 0;
        sprintf('Forcing reaction %s "not low" is infeasible',model.rxns{r})     
    else
        warning(sprintf('Problem forcing reaction %s "not low": %s',model.rxns{r},output_.cplexstatusstring)); 
    end
    
    % Evaluate direction (high=1, low=-1)
    if (x(r)>y(r))
        highOrLow(r) = -1;
        confidence(r) = x(r)-y(r);
        use = 'always low';
    elseif (x(r)<y(r))
        highOrLow(r) = 1;
        confidence(r) = y(r)-x(r);
        use = 'never low';
    else
        use = 'undetermined';
    end
    sprintf('Reaction %s found to be %s with confidence=%i',...
        model.rxns{r},use,confidence(r))

end